/* Common definitions for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#if !defined(__GPASM_H)
#define __GPASM_H

#include <stdio.h>
#include "processor.h"
#include "i_memory.h"

//#define GPASM_VERSION_STRING "gpasm 0.0.8a alpha"
#define GPASM_VERSION_STRING ("gpasm-" VERSION " alpha")

#define SPECIAL_PATH	DATADIR "/special.inc"

/* special.inc has macros for the 16bit core and everything else.
 * This symbol will get placed into the symbol table for the 16bit cores
 * and thus allow compile-time selection of the proper macro set */
#define __16bit_core_	"__16bit_core_"

#define STRCMP(s1, s2)	(state.case_insensitive ? \
			 strcasecmp((s1), (s2)) : \
			 strcmp((s1), (s2)))
			 
typedef int gpasmVal; 		/* The type that internal arithmetic uses */
enum gpasmValTypes {gvt_constant, gvt_cblock, gvt_org, gvt_address};

enum outfile { normal, suppress, named };
enum file_types { ft_src, ft_hex, ft_lst, ft_cod, ft_other }; /* allowed file types */

/************************************************************************
 * CONFIG addresses for the 18cxxx parts
 */

#define CONFIG1L  0x300000
#define CONFIG1H  0x300001
#define CONFIG2L  0x300002
#define CONFIG2H  0x300003
#define CONFIG3H  0x300005
#define CONFIG4L  0x300006
#define DEVID1    0x3ffffe
#define DEVID2    0x3fffff

// Config address for everything else:
#define CONFIG_ADDRESS_14  0x2007
#define CONFIG_ADDRESS_12  0x0fff


extern struct gpasm_state {
  int radix;
  enum { inhx8m, inhx8s, inhx32 } hex_format;
  int case_insensitive;
  int quiet;
  enum pic_processor processor;
  struct px *processor_info;    /* Processor identifiers (e.g. name) */

  int pass;					/* 1 or 2 */
  int processor_chosen;		/* Nonzero after processor-specific init */
  struct symbol_table
  *stBuiltin,			/* Built-ins: instructions, pseudo-ops */
    *stGlobal,			/* Global symbols */
    *stTop,			/* Top of locals stack (stGlobal is base) */
    *stDefines,			/* Preprocessor #defines */
    *stMacros;			/* Macros */
  MemBlock *i_memory;		/* Instruction memory linked list */
  //  unsigned int
  //    i_memory[MAX_I_MEM];
  unsigned int
  //  c_memory[MAX_C_MEM],	/* Configuration memory */
  c_memory_base;		/* Base address of configuration memory */
  char
    badram[MAX_RAM];		/* nonzero indicates illegal memory */
  unsigned int maxram;		/* Highest legal memory location */
  unsigned int org;		/* Current code-generation point */
  enum outfile
    lstfile,			/* List output file control */
    hexfile,			/* Hex output file control */
    codfile;			/* Symbol output file control */
  char *srcfilename,		/* Source (.asm) file name */
    basefilename[BUFSIZ],	/* basename for generating hex,list,symbol filenames */
    hexfilename[BUFSIZ],	/* Hex (.hex) file name */
    lstfilename[BUFSIZ],	/* List (.lst) file name */
    codfilename[BUFSIZ];	/* Symbol (.cod) file name */
  struct {			/* List file state: */
    FILE *f;			/*   List file output */
    unsigned int
      lineofpage,		/*   What line are we at within the page */
      page,			/*   What page are we at */
      linesperpage,		/*   Lines per page */
      line_number,              /*   What line are we at within the file */
      symboltable;		/*   Symbol table dump enabled */
    struct {
      unsigned int was_org;	/*   value of state.org at start of line */
		    		/*   What kind of line was it? */
      enum { none,		/*     Nothing - blank line */
	     org,		/*     ORG pseudo-op */
	     insn,		/*     Some other instruction or pseudo */
	     equ,		/*     An equate */
	     set,		/*     A SET or '=' */
	     config }           /*     A __config line */
        linetype;
    } line;
    char startdate[80];		/*   When assembly started */
    int enabled;		/*   nonzero if listing is enabled */
    char title_name[80];	/*   given in TITLE directive */
    int tabstop;		/*   tab-stop distance */
    struct file_context *fc;    /*   Position in the file context stack */
  } lst;
  struct {			/* Symbol file state: */
    FILE *f;			/*   Symbol file output */
    int enabled;		/*   nonzero if symbol file is enabled */
    int emitting;               /*   flag indicating when an opcode is emitted */
  } cod;
  struct source_context *src;	/* Top of the stack of source files */
  struct file_context *files;   /* Top of the stack of all files */
  struct {			/* Totals for errors, warnings, messages */
    int errors;
    int warnings;
    int messages;
  } num;
  struct amode *astack;		/* Stack of amodes (macros, etc) */
  gpasmVal cblock;		/* cblock constant */
  struct macro_head *mac_head;  /* Starting a macro... */
  struct macro_body **mac_prev; /* Stitching ptr */
  struct macro_body *mac_body;	/* While we're building a macro */
  struct macro_head *while_head;/* WHILEs work a lot like macros... */
} state;

#define MEM_USED_MASK 0x80000000 /* Means occupied in state.i_memory above */

struct variable {
  int value;
  enum gpasmValTypes type;
};

#define MAX_INCLUDE_PATHS 100
extern char *include_paths[];
extern int n_include_paths;

/************************************************************************/

/* Parse node: created by the parser, interpreted by the 'backend' */

struct pnode {
  enum pnode_tag { constant, symbol, string, list, binop, unop } tag;
  union {
    int constant;
    char *symbol;
    struct {
      struct pnode *head, *tail;
    } list;
    struct {
      int op;
      struct pnode *p0, *p1;
    } binop;
    struct {
      int op;
      struct pnode *p0;
    } unop;
    char *string;
  } value;
};

/************************************************************************/

/* file_context: a structure to keep track of all files that have been
		 opened.  Used to create the list of project files
		 that can be found in the .cod file.  */

struct file_context {
  char *name;                     /* file name */
  unsigned int id;                /* Unique identifier */
  enum file_types ft;             /* allowed file types */
  struct file_context *prev;      /* Singly-linked list pointer */
};

struct source_context {
  char *name;
  FILE *f, *f2;
  struct yy_buffer_state *yybuf;
  unsigned int line_number;
  struct source_context *prev;
  struct file_context *fc;    /*   Position in the file context stack */
  char line[BUFSIZ];
};

enum globalLife { TEMPORARY, PERMANENT };

void yyerror(char *s);
int gpasm_number(char *);
void open_src(char *name, int is_include_file);
gpasmVal do_or_append_insn(char *op, struct pnode *parms);
void set_global(char *name,
		gpasmVal value,
		enum globalLife lifetime,
		enum gpasmValTypes type);
void select_radix(char *name);
struct file_context *add_file(unsigned int type, char *name);
void free_files(void);
void macro_append(void);

/************************************************************************/

struct macro_head {
  int pass;			/* Pass in which macro was defined: 1 or 2 */
  struct pnode *parms;
  struct macro_body *body;
  int line_number;
};

struct macro_body {
  char *label;			/* Label for the line */
  char *op;			/* Operation (or NULL) */
  struct pnode *parms;		/* Parameters for op (or NULL) */
  char *src_line;		/* Original source line - for listing */
  struct macro_body *next;	/* Next line in listing */
};

#endif
