/* Error handling for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "gperror.h"
#include "lst.h"




char *gp_geterror(unsigned int code)
{

  switch(code) {
  case GPE_RANGE:
    return "address is out of range.";
    break;
  case GPE_IHEX:
    return "Code is too big for hex file.";
    break;
  GPE_UNKNOWN:
  default:
    return "UNKNOWN";
  }
}

void gperror(unsigned int code,
	     char *message)
{
  char full_message[BUFSIZ];

  if (state.pass == 2) {
    if(message == NULL)
      message = gp_geterror(code);

    if (state.src)
      sprintf(full_message,
	      "Error %s %d : %03d %s",
	      state.src->name,
	      state.src->line_number,
	      code,
	      message);
    else
      sprintf(full_message,
	      "Error %03d %s",
	      code,
	      message);
    printf("%s\n", full_message);
    lst_line(full_message);

    state.num.errors++;
  }
}

char *gp_getwarning(unsigned int code)
{
  switch(code) {
  case GPW_RANGE:
    return "Out of range. Least significant bits used.";
    break;
  case GPW_EXPECTED:
    return "expected";
  case GPW_UNKNOWN:
  default:
    return "UNKNOWN";
  }
}

void gpwarning(unsigned int code,
	       char *message)
{
  char full_message[BUFSIZ];

  if (state.pass == 2) {
    if(message == NULL)
      message = gp_getwarning(code);

    if (state.src)
      sprintf(full_message,
	      "Warning %s %d : %03d %s",
	      state.src->name,
	      state.src->line_number,
	      code,
	      message);
    else
      sprintf(full_message,
	      "Warning %03d %s",
	      code,
	      message);
    printf("%s\n", full_message);
    lst_line(full_message);

    state.num.warnings++;
  }
}


char *gp_getmessage(unsigned int code)
{
  switch(code) {
  case GPM_USER:
    return "MESSAGE:";
  case GPM_BANK:
    return "Argument out of range. Least significant bits used.";
  case GPM_RANGE:
    return "Program word too large. Truncated to core size.";
  case GPM_IDLOC:
    return "ID Locations value too large. Last four hex digits used.";
  case GPM_NOF:
    return "Using default destination of 1 (file).";
  case GPM_PAGE:
    return "Crossing page boundary - ensure page bits are set.";
  case GPM_PAGEBITS:
    return "Setting page bits.";
  case GPM_SUPVAL:
    return "Warning level superceded by command line value.";
  case GPM_SUPLIN:
    return "Macro expansion superceded by command line value.";
  case GPM_SUPRAM:
    return "Superceding current maximum RAM and RAM map.";
  case GPM_EXTPAGE:
    return "Page bits are not needed.";
  case GPM_CBLOCK:
    return "CBLOCK will begin at address 0.";
  case GPM_UNKNOWN:
  default:
    return "UNKNOWN MESSAGE";
  }
};

void gpmessage(unsigned int code,
	       char *message)
{
  char full_message[BUFSIZ];

  if(message == NULL)
    message = gp_getmessage(code);


  if (state.pass == 2) {
    if (state.src)
      sprintf(full_message,
	      "Message %s %d : %03d %s",
	      state.src->name,
	      state.src->line_number,
	      code,
	      message);
    else
      sprintf(full_message,
	      "Message %03d %s",
	      code,
	      message);
    printf("%s\n", full_message);
    lst_line(full_message);

    state.num.messages++;
  }
}

