/* ".HEX" file output for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "hex.h"
#include "gperror.h"

static int sum;
static FILE *hex;

static void new_record()
{
  fprintf(hex, ":");
  sum = 0;
}

static void write_byte(int b)
{
  sum += b;
    
  assert((0 <= b) && (b <= 255));
  fprintf(hex, "%02X", b);
}

/* Write big-endian word */
static void write_bg_word(int w)
{
  write_byte((w >> 8) & 0xff);
  write_byte(w & 0xff);
}

/* Write little-endian word */
static void write_word(int w)
{
  write_byte(w & 0xff);
  write_byte((w >> 8) & 0xff);
}

static void end_record()
{
  write_byte((-sum) & 0xff);
  fprintf(hex, "\n");
}

#define MAX_IHEX16 0x8000

static void write_i_mem(void)
{
  MemBlock *m = state.i_memory;
  int i, j;
  int size_error = 0;

  while(m) {
    i = m->base << I_MEM_BITS;

    if( (i >= MAX_IHEX16) && !size_error) {
      gperror(GPE_IHEX,NULL);
      fprintf(stderr,"(you probably have __config data for an 18cxxx device.\n");
      fprintf(stderr,"If gpasm supported INH32 files, this error wouldn't be here.)\n");
      size_error = 1;
    }
      
    while (i < MAX_IHEX16) {
      if ((i_memory_get(i) & MEM_USED_MASK) == 0) {
	++i;
      } else {
	j = i;
	while ((i_memory_get(i) & MEM_USED_MASK)) {
	  ++i;
	  if ((i & 7) == 0)
	    break;
	}
	/* Now we have a run of (i - j) occupied memory locations. */
	new_record();
	write_byte(2 * (i - j));
	write_bg_word(2 * j);
	write_byte(0);
	while (j < i)
	  write_word(i_memory_get(j++));
	end_record();
      }
    }
    m = m->next;
  }
}

/*************************************************************
 *
 * write_c_mem - write configuration information (as specified
 *               by the __CONFIG directive) to the hex file.
 *
 * NOTE:
 * 1) This is currently only used for the 18cxxx devices (the
 *    others store the config data in i_memory[]).
 * 2) There is an intentional bug. The hex file is limited to
 *    only 64k bytes, but the 18cxxx config data begins at
 *    address 0x300000. So for now, gpasm will place the 
 *    config data at address 0xff00 - 0xffff. The algorithm 
 *    for mapping from 0x300000 to 0xff00 is:
 *
 *    new_cfg_addr = 0xff00 | (old_cfg & 0x00ff);
 */
static void write_c_mem(void)
{

  int i, j;

  i = CONFIG1L; 
  while(i<=CONFIG4L) {

    if ((i_memory_get(i) & MEM_USED_MASK) == 0) { 
      ++i;
    } else {
      j = i;
      while ((i_memory_get(i) & MEM_USED_MASK)) {
	++i;
	if ((i & 7) == 0)
	  break;
      }
      /* Now we have a run of (i - j) occupied memory locations. */
      new_record();
      write_byte(2 * (i - j));
      //write_bg_word(2 * (j + state.c_memory_base));
      write_bg_word(2 * (0x7f00 | j&0xff));
      write_byte(0);
      while (j < i)
	write_word(i_memory_get(j++));
      end_record();
    }
  }

  j = 0;
  if(i_memory_get(DEVID1)  & MEM_USED_MASK) {
     j = DEVID1;
     if(i_memory_get(DEVID2)  & MEM_USED_MASK)
       i = DEVID2+1;
     else
       i = DEVID1+1;
  } else if(i_memory_get(DEVID2)  & MEM_USED_MASK) {
     j = DEVID2;
     i = DEVID2+1;
  }
  if(j) {
    new_record();
    write_byte(2 * (i - j));
    //write_bg_word(2 * (j + state.c_memory_base));
    write_bg_word(2 * (0x7f00 | j&0xff));
    write_byte(0);
    while (j < i)
	write_word(i_memory_get(j++));
    end_record();
  }

}



void dump_hex()
{
  char *pc;

  switch (state.hexfile) {
  case normal:
    if(state.hexfilename[0] == '\0') {
      strcpy(state.hexfilename, state.basefilename);
      strcat(state.hexfilename, ".hex");
    }
    break;
  case suppress:
    return;
  case named:
    /* Don't need to do anything - name is already set up */
    ;
  }

  if (state.num.errors) {
    /* Remove the hex file (if any) */
    unlink(state.hexfilename);
    return;
  }

  /* No error: overwrite the hex file */

  hex = fopen(state.hexfilename, "wt");

  write_i_mem();
  write_c_mem();
  new_record();
  write_byte(0);
  write_word(0);
  write_byte(1);
  end_record();

  fclose(hex);
}
